<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\AdminController;
use App\Http\Controllers\AuthController;
use App\Http\Controllers\WorkspaceController;
use App\Http\Controllers\PageController;
use App\Http\Controllers\ProductController;
use App\Http\Controllers\CustomerController;
use App\Http\Controllers\OrderController;
use App\Http\Controllers\CrmCardController;
use App\Http\Controllers\MediaController;
use App\Http\Controllers\SettingsController;
use App\Http\Controllers\DataManagementController;
use App\Http\Controllers\PaymentController;
// POS Controllers
use App\Http\Controllers\PosCustomerController;
use App\Http\Controllers\PosProductController;
use App\Http\Controllers\PosChannelController;
use App\Http\Controllers\PosPaymentController;
use App\Http\Controllers\PosAdminController;
use App\Http\Controllers\PosSaleController;
use App\Http\Controllers\AffiliateController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

// Auth Routes (Public)
Route::middleware(['cors'])->group(function () {
    Route::post('/register', [AuthController::class, 'register']);
    Route::post('/login', [AuthController::class, 'login']);
    Route::get('/me', [AuthController::class, 'user'])->middleware('auth:api');
    Route::post('/logout', [AuthController::class, 'logout'])->middleware('auth:api');
});

// Test endpoint (no auth required)
Route::get('/test', function () {
    return response()->json(['message' => 'API connection successful!', 'status' => 'ok']);
});

// Login route for API (returns 401 for unauthenticated requests)
Route::get('/login', function () {
    return response()->json(['message' => 'Unauthenticated.'], 401);
})->name('login');

// Public Routes (No Authentication Required)
Route::middleware(['web', 'cors'])->group(function () {


    // Global settings - Public for all users
    Route::get('/global-settings', [AdminController::class, 'getGlobalSettings']);

    // Published pages - Public for viewing
    Route::get('/pages/published/{slug}', [PageController::class, 'getPublishedPage']);
});

// Protected Routes (Require Authentication)
Route::middleware(['cors', 'auth:api'])->group(function () {
    Route::post('/refresh', [AuthController::class, 'refresh']);
    Route::post('/logout', [AuthController::class, 'logout']);
    Route::get('/user', [AuthController::class, 'user']);
    Route::put('/profile', [AuthController::class, 'updateProfile']);

    // Workspace initialization
    Route::get('/workspace', [WorkspaceController::class, 'index']);

    // API Resources CRUD - Protected for viewing and editing
    Route::get('/pages', [PageController::class, 'index']);
    Route::get('/pages/{page}', [PageController::class, 'show']);
    Route::post('/pages', [PageController::class, 'store']);
    Route::put('/pages/{page}', [PageController::class, 'update']);
    Route::delete('/pages/{page}', [PageController::class, 'destroy']);
    Route::post('/pages/{page}/duplicate', [PageController::class, 'duplicate']);
    Route::post('/pages/{page}/publish', [PageController::class, 'publish']);
    Route::post('/pages/{page}/unpublish', [PageController::class, 'unpublish']);




    Route::apiResource('products', ProductController::class);
    Route::apiResource('customers', CustomerController::class);
    Route::apiResource('orders', OrderController::class);

    // CRM
    Route::get('/crm-cards', [CrmCardController::class, 'index']);
    Route::put('/crm-cards/{crm_card}/status', [CrmCardController::class, 'updateStatus']);

    // Media (File Uploads)
    Route::get('/media', [MediaController::class, 'index']);
    Route::post('/media', [MediaController::class, 'store']);
    Route::delete('/media/{media}', [MediaController::class, 'destroy']);

    // Settings
    Route::get('/settings', [SettingsController::class, 'show']);
    Route::put('/settings', [SettingsController::class, 'update']);

    // Data Management
    Route::post('/import', [DataManagementController::class, 'import']);

    // Cleanup pro features
    Route::post('/cleanup-pro', [PageController::class, 'cleanupPro']);

    // Payment routes
    Route::post('/payments/create', [PaymentController::class, 'createPayment']);
    Route::post('/payments/webhook', [PaymentController::class, 'handleWebhook']);
    Route::get('/payments/{paymentId}/status', [PaymentController::class, 'checkStatus']);
    Route::get('/plans', [PaymentController::class, 'getPlans']);
    Route::get('/subscriptions', [PaymentController::class, 'getSubscriptions']);
    Route::post('/payments/validate-referral', [PaymentController::class, 'validateReferral']);

    // POS Auth
    Route::middleware('cors')->group(function () {
        Route::post('/pos/register', [AuthController::class, 'register']);
        Route::post('/pos/login', [AuthController::class, 'login']);

        Route::middleware('auth:api')->group(function () {
            Route::post('/pos/logout', [AuthController::class, 'logout']);
            Route::get('/pos/user', [AuthController::class, 'user']);
        });
    });

    // POS CRUD Routes (Protected with JWT)
    Route::middleware('cors')->group(function () {
        Route::apiResource('pos-customers', PosCustomerController::class);
        Route::apiResource('pos-products', PosProductController::class);
        Route::apiResource('pos-channels', PosChannelController::class);
        Route::apiResource('pos-payments', PosPaymentController::class);
        Route::apiResource('pos-admins', PosAdminController::class);
    Route::apiResource('pos-sales', PosSaleController::class);
    });

    // Affiliate routes
    Route::get('/affiliate', [AffiliateController::class, 'getAffiliateData']);
    Route::post('/affiliate/withdraw', [AffiliateController::class, 'withdraw']);
    Route::put('/affiliate/code', [AffiliateController::class, 'updateAffiliateCode']);

    // Admin Routes (Require Authentication and Admin Role)
    Route::middleware(['cors', 'auth:api', 'admin'])->prefix('admin')->group(function () {
        Route::get('/dashboard', [AdminController::class, 'dashboard']);
        Route::get('/users', [AdminController::class, 'indexUsers']);
        Route::get('/users/{id}', [AdminController::class, 'showUser']);
        Route::put('/users/{id}', [AdminController::class, 'updateUser']);
        Route::delete('/users/{id}', [AdminController::class, 'destroyUser']);
        Route::get('/transactions', [AdminController::class, 'indexTransactions']);
        Route::get('/transactions/{id}', [AdminController::class, 'showTransaction']);
        Route::get('/subscriptions', [AdminController::class, 'indexSubscriptions']);
        Route::get('/subscriptions/stats', [AdminController::class, 'subscriptionStats']);
        Route::get('/subscriptions/{id}', [AdminController::class, 'showSubscription']);
        Route::get('/user-growth', [AdminController::class, 'userGrowthStats']);
        Route::get('/revenue-growth', [AdminController::class, 'revenueGrowthStats']);
        Route::get('/affiliate-settings', [AdminController::class, 'getAffiliateSettings']);
        Route::put('/affiliate-settings', [AdminController::class, 'updateAffiliateSettings']);
        Route::get('/affiliate-withdraws', [AdminController::class, 'getAffiliateWithdraws']);
        Route::post('/affiliate-withdraws/{id}/approve', [AdminController::class, 'approveWithdraw']);
        Route::post('/affiliate-withdraws/{id}/reject', [AdminController::class, 'rejectWithdraw']);

        // Plan management
        Route::get('/plans', [AdminController::class, 'getPlans']);
        Route::put('/plans/{id}', [AdminController::class, 'updatePlan']);

        // Global settings
        Route::get('/global-settings', [AdminController::class, 'getGlobalSettings']);
        Route::put('/global-settings', [AdminController::class, 'updateGlobalSettings']);
    });
});
