<?php

namespace App\Http\Controllers;

use App\Models\Product;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $products = Product::where('user_id', $request->user()->id)
            ->with('media')
            ->get()
            ->map(function ($product) use ($request) {
                $product->image_url = $product->media ? $request->getSchemeAndHttpHost() . '/' . $product->media->path : $product->image_url;
                return $product;
            });
        return response()->json($products);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'price' => 'required|numeric|min:0',
            'sku' => 'nullable|string|max:255|unique:products',
            'category' => 'nullable|string|max:255',
            'media_id' => 'nullable|exists:media,id',
            'image_url' => 'nullable|string',
            'stock' => 'required|integer|min:0',
        ]);

        $product = Product::create([
            'user_id' => $request->user()->id,
            'name' => $request->name,
            'price' => $request->price,
            'sku' => $request->sku,
            'category' => $request->category,
            'media_id' => $request->media_id,
            'image_url' => $request->image_url,
            'stock' => $request->stock,
        ]);

        // Load media relationship and set image_url if media exists
        $product->load('media');
        if ($product->media) {
            $product->image_url = $request->getSchemeAndHttpHost() . '/' . $product->media->path;
        }

        return response()->json($product, 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, string $id)
    {
        $product = Product::where('user_id', $request->user()->id)
            ->where('id', $id)
            ->firstOrFail();

        return response()->json($product);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'price' => 'sometimes|required|numeric|min:0',
            'sku' => 'sometimes|nullable|string|max:255|unique:products,sku,' . $id,
            'category' => 'nullable|string|max:255',
            'media_id' => 'nullable|exists:media,id',
            'image_url' => 'nullable|string',
            'stock' => 'sometimes|required|integer|min:0',
        ]);

        $product = Product::where('user_id', $request->user()->id)
            ->where('id', $id)
            ->firstOrFail();

        $product->update($request->only([
            'name', 'price', 'sku', 'category', 'media_id', 'image_url', 'stock'
        ]));

        // Load media relationship and set image_url if media exists
        $product->load('media');
        if ($product->media) {
            $product->image_url = $request->getSchemeAndHttpHost() . '/' . $product->media->path;
        }

        return response()->json($product);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, string $id)
    {
        $product = Product::where('user_id', $request->user()->id)
            ->where('id', $id)
            ->firstOrFail();

        $product->delete();

        return response()->json(['message' => 'Product deleted successfully']);
    }
}
