<?php

namespace App\Http\Controllers;

use App\Models\Page;
use App\Models\Block;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PageController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $pages = Page::where('user_id', auth()->id())
            ->with(['blocks' => function ($query) {
                $query->orderBy('order');
            }])
            ->get()
            ->map(function ($page) {
                return [
                    'id' => $page->id,
                    'name' => $page->name,
                    'published' => $page->published,
                    'slug' => $page->slug,
                    'content' => $page->blocks->map(function ($block) {
                        // Convert props to string to perform replacement
                        $propsString = json_encode($block->props);

                        // Replace local development URL with production URL
                        $newPropsString = str_replace(
                            'https://api.initoko.online',
                            'https://api.initoko.online',
                            $propsString
                        );

                        return [
                            'id' => $block->id,
                            'type' => $block->type,
                            'props' => json_decode($newPropsString, true),
                        ];
                    })->toArray(),
                    'created_at' => $page->created_at,
                    'updated_at' => $page->updated_at,
                ];
            });

        return response()->json($pages);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'content' => 'sometimes|array', // Changed from 'blocks' to 'content'
            'content.*.type' => 'required|string',
            'content.*.props' => 'required|array',
            'content.*.order' => 'required|integer',
        ]);

        DB::beginTransaction();

        try {
            $page = Page::create([
                'user_id' => auth()->id(),
                'name' => $request->name,
            ]);

            if ($request->has('content')) { // Changed from 'blocks' to 'content'
                foreach ($request->content as $blockData) { // Changed from $request->blocks to $request->content
                    $page->blocks()->create([
                        'type' => $blockData['type'],
                        'props' => $blockData['props'],
                        'order' => $blockData['order'],
                    ]);
                }
            }

            $page->load(['blocks' => function ($query) {
                $query->orderBy('order');
            }]);

            DB::commit();

        return response()->json([
            'id' => $page->id,
            'name' => $page->name,
            'content' => $page->blocks->map(function ($block) {
                return [
                    'id' => $block->id,
                    'type' => $block->type,
                    'props' => $block->props,
                ];
            })->toArray(),
            'created_at' => $page->created_at,
            'updated_at' => $page->updated_at,
        ], 201);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(['error' => 'Failed to create page'], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, string $id)
    {
        $page = Page::where('id', $id)
            ->where('user_id', auth()->id())
            ->with(['blocks' => function ($query) {
                $query->orderBy('order');
            }])
            ->firstOrFail();

        return response()->json([
            'id' => $page->id,
            'name' => $page->name,
            'content' => $page->blocks->map(function ($block) {
                return [
                    'id' => $block->id,
                    'type' => $block->type,
                    'props' => $block->props,
                ];
            })->toArray(),
            'created_at' => $page->created_at,
            'updated_at' => $page->updated_at,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'content' => 'sometimes|array', // Changed from 'blocks' to 'content'
            'content.*.type' => 'required|string',
            'content.*.props' => 'required|array',
            'content.*.order' => 'required|integer',
        ]);

        $page = Page::where('id', $id)
            ->where('user_id', auth()->id())
            ->firstOrFail();

        DB::beginTransaction();

        try {
            $page->update($request->only(['name']));

            if ($request->has('content')) { // Changed from 'blocks' to 'content'
                // Delete existing blocks
                $page->blocks()->delete();

                // Create new blocks
                foreach ($request->content as $blockData) { // Changed from $request->blocks to $request->content
                    $page->blocks()->create([
                        'type' => $blockData['type'],
                        'props' => $blockData['props'],
                        'order' => $blockData['order'],
                    ]);
                }

                \Log::info('Blocks created successfully', [
                    'page_id' => $page->id,
                    'blocks_count' => count($request->content),
                    'blocks_data' => $request->content
                ]);
            }

            $page->load(['blocks' => function ($query) {
                $query->orderBy('order');
            }]);

            \Log::info('Page updated successfully', [
                'page_id' => $page->id,
                'page_name' => $page->name,
                'blocks_count' => $page->blocks->count()
            ]);

            DB::commit();

            return response()->json([
                'id' => $page->id,
                'name' => $page->name,
                'content' => $page->blocks->map(function ($block) {
                    return [
                        'id' => $block->id,
                        'type' => $block->type,
                        'props' => $block->props,
                    ];
                })->toArray(),
                'created_at' => $page->created_at,
                'updated_at' => $page->updated_at,
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(['error' => 'Failed to update page'], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, string $id)
    {
        $page = Page::where('id', $id)
            ->where('user_id', auth()->id())
            ->firstOrFail();

        $page->delete();

        return response()->json(['message' => 'Page deleted successfully']);
    }

    /**
     * Duplicate a page
     */
    public function duplicate(Request $request, string $id)
    {
        $originalPage = Page::where('id', $id)
            ->where('user_id', auth()->id())
            ->with(['blocks' => function ($query) {
                $query->orderBy('order');
            }])
            ->firstOrFail();

        DB::beginTransaction();

        try {
            $newPage = Page::create([
                'user_id' => auth()->id(),
                'name' => $originalPage->name . ' (Copy)',
            ]);

            foreach ($originalPage->blocks as $block) {
                $newPage->blocks()->create([
                    'type' => $block->type,
                    'props' => $block->props,
                    'order' => $block->order,
                ]);
            }

            $newPage->load(['blocks' => function ($query) {
                $query->orderBy('order');
            }]);

            DB::commit();

            return response()->json([
                'id' => $newPage->id,
                'name' => $newPage->name,
                'content' => $newPage->blocks->map(function ($block) {
                    return [
                        'id' => $block->id,
                        'type' => $block->type,
                        'props' => $block->props,
                    ];
                })->toArray(),
                'created_at' => $newPage->created_at,
                'updated_at' => $newPage->updated_at,
            ], 201);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(['error' => 'Failed to duplicate page'], 500);
        }
    }



    /**
     * Publish a page
     */
    public function publish(Request $request, string $id)
    {
        $request->validate([
            'slug' => 'required|string|max:255|unique:pages,slug,' . $id,
        ]);

        $page = Page::where('id', $id)
            ->where('user_id', auth()->id())
            ->firstOrFail();

        $page->update([
            'published' => true,
            'slug' => $request->slug,
        ]);

        return response()->json([
            'message' => 'Page published successfully',
            'page' => [
                'id' => $page->id,
                'name' => $page->name,
                'published' => $page->published,
                'slug' => $page->slug,
            ]
        ]);
    }

    /**
     * Unpublish a page
     */
    public function unpublish(Request $request, string $id)
    {
        $page = Page::where('id', $id)
            ->where('user_id', auth()->id())
            ->firstOrFail();

        $page->update([
            'published' => false,
        ]);

        return response()->json([
            'message' => 'Page unpublished successfully',
            'page' => [
                'id' => $page->id,
                'name' => $page->name,
                'published' => $page->published,
                'slug' => $page->slug,
            ]
        ]);
    }

    /**
     * Get published page by slug (public route)
     */
    public function getPublishedPage(Request $request, string $slug)
    {
        $page = Page::where('slug', $slug)
            ->where('published', true)
            ->with(['blocks' => function ($query) {
                $query->orderBy('order');
            }, 'user.subscriptions']) // Eager load user and subscriptions
            ->firstOrFail();

        // Determine if the page owner has an active pro subscription
        $isPro = $page->user ? $page->user->subscriptions()->where('status', 'active')->exists() : false;

        return response()->json([
            'id' => $page->id,
            'name' => $page->name,
            'slug' => $page->slug,
            'theme' => $page->theme, // Add theme to the response
            'isPro' => $isPro,       // Add isPro status to the response
            'content' => $page->blocks->map(function ($block) {
                // Convert props to string to perform replacement
                $propsString = json_encode($block->props);

                // Replace local development URL with production URL
                $newPropsString = str_replace(
                    'https://api.initoko.online',
                    'https://api.initoko.online',
                    $propsString
                );

                return [
                    'id' => $block->id,
                    'type' => $block->type,
                    'props' => json_decode($newPropsString, true),
                ];
            })->toArray(),
            'created_at' => $page->created_at,
            'updated_at' => $page->updated_at,
        ]);
    }

    /**
     * Cleanup pro features for users without active subscriptions
     */
    public function cleanupPro(Request $request)
    {
        $user = auth()->user();

        // Check if user has active pro subscription
        $activeSubscription = $user->subscriptions()->where('status', 'active')->first();

        if ($activeSubscription) {
            return response()->json(['message' => 'User has active pro subscription, no cleanup needed']);
        }

        // Define pro-only block types
        $proOnlyBlockTypes = ['pricing', 'video', 'countdown'];

        DB::beginTransaction();

        try {
            // Get all pages for the user
            $pages = Page::where('user_id', $user->id)->with('blocks')->get();

            $totalBlocksRemoved = 0;

            foreach ($pages as $page) {
                // Find pro-only blocks to remove
                $blocksToRemove = $page->blocks->filter(function ($block) use ($proOnlyBlockTypes) {
                    return in_array(strtolower($block->type), $proOnlyBlockTypes);
                });

                if ($blocksToRemove->count() > 0) {
                    // Delete pro-only blocks
                    $page->blocks()->whereIn('id', $blocksToRemove->pluck('id'))->delete();
                    $totalBlocksRemoved += $blocksToRemove->count();

                    \Log::info('Removed pro-only blocks from page', [
                        'page_id' => $page->id,
                        'page_name' => $page->name,
                        'blocks_removed' => $blocksToRemove->count(),
                        'block_types' => $blocksToRemove->pluck('type')->toArray()
                    ]);
                }
            }

            DB::commit();

            return response()->json([
                'message' => 'Pro features cleaned up successfully',
                'total_blocks_removed' => $totalBlocksRemoved,
                'pages_affected' => $pages->count()
            ]);

        } catch (\Exception $e) {
            DB::rollback();
            \Log::error('Failed to cleanup pro features', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);
            return response()->json(['error' => 'Failed to cleanup pro features'], 500);
        }
    }
}
