<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\CrmCard;
use Illuminate\Http\Request;

class CustomerController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $customers = Customer::where('user_id', auth()->id())->get();
        return response()->json($customers);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => ['nullable', 'string', 'max:20', 'regex:/^\d+$/'],
            'email' => ['nullable', 'email', 'max:255', 'regex:/@/'],
            'notes' => 'nullable|string',
        ]);

        $customer = Customer::create([
            'user_id' => auth()->id(),
            'name' => $request->name,
            'phone' => $request->phone,
            'email' => $request->email,
            'notes' => $request->notes,
        ]);

        // Create a CRM card for the new customer
        CrmCard::create([
            'user_id' => auth()->id(),
            'customer_id' => $customer->id,
            'status' => 'To Contact',
            'value' => 0,
        ]);

        return response()->json($customer, 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, string $id)
    {
        $customer = Customer::where('user_id', auth()->id())
            ->where('id', $id)
            ->firstOrFail();

        return response()->json($customer);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'phone' => ['nullable', 'string', 'max:20', 'regex:/^\d+$/'],
            'email' => ['nullable', 'email', 'max:255', 'regex:/@/'],
            'notes' => 'nullable|string',
        ]);

        $customer = Customer::where('user_id', auth()->id())
            ->where('id', $id)
            ->firstOrFail();

        $customer->update($request->only(['name', 'phone', 'email', 'notes']));

        return response()->json($customer);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, string $id)
    {
        $customer = Customer::where('user_id', auth()->id())
            ->where('id', $id)
            ->firstOrFail();

        $customer->delete();

        return response()->json(['message' => 'Customer deleted successfully']);
    }
}
