<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\AffiliateWithdraw;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class AffiliateController extends Controller
{
    /**
     * Get affiliate data for authenticated user
     */
    public function getAffiliateData(Request $request)
    {
        $user = $request->user();

        if (!$user->is_affiliate) {
            return response()->json(['message' => 'User is not an affiliate'], 403);
        }

        $withdraws = $user->affiliateWithdraws()->orderBy('created_at', 'desc')->get();

        $referredUsers = $user->referredUsers()->select('id', 'name', 'email', 'created_at')->get();

        return response()->json([
            'is_affiliate' => true,
            'affiliate_code' => $user->affiliate_code,
            'affiliate_balance' => $user->affiliate_balance,
            'total_referred' => $user->total_referred,
            'withdraws' => $withdraws,
            'referred_users' => $referredUsers,
        ]);
    }

    /**
     * Update affiliate code
     */
    public function updateAffiliateCode(Request $request)
    {
        $user = $request->user();

        if (!$user->is_affiliate) {
            return response()->json(['message' => 'User is not an affiliate'], 403);
        }

        $request->validate([
            'affiliate_code' => 'required|string|size:8|unique:users,affiliate_code,' . $user->id,
        ]);

        $user->update([
            'affiliate_code' => Str::upper($request->affiliate_code)
        ]);

        return response()->json([
            'message' => 'Affiliate code updated successfully',
            'affiliate_code' => $user->affiliate_code,
        ]);
    }

    /**
     * Request a withdrawal
     */
    public function withdraw(Request $request)
    {
        $user = $request->user();

        if (!$user->is_affiliate) {
            return response()->json(['message' => 'User is not an affiliate'], 403);
        }

        $validator = Validator::make($request->all(), [
            'amount' => 'required|integer|min:50000|max:' . $user->affiliate_balance,
            'account_name' => 'required|string|max:255',
            'account_number' => 'required|string|max:255',
            'bank_name' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $withdraw = AffiliateWithdraw::create([
            'user_id' => $user->id,
            'amount' => $request->amount,
            'account_name' => $request->account_name,
            'account_number' => $request->account_number,
            'bank_name' => $request->bank_name,
            'status' => 'pending',
        ]);

        // Kurangi saldo affiliate saat request withdrawal
        $user->decrement('affiliate_balance', $request->amount);

        return response()->json([
            'message' => 'Withdrawal request submitted successfully',
            'withdraw' => $withdraw,
        ], 201);
    }
}
