<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\PosSale;
use App\Models\Subscription;
use App\Models\Payment;
use App\Models\AffiliateSetting;
use App\Models\AffiliateWithdraw;
use App\Models\GlobalSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class AdminController extends Controller
{
    /**
     * Get dashboard stats
     */
    public function dashboard()
    {
        $stats = [
            'total_users' => User::count(),
            'total_transactions' => Payment::count(),
            'total_subscriptions' => Subscription::where('status', 'active')->count(),
            'total_revenue' => Payment::sum('amount') ?? 0,
            'total_pos_sales' => PosSale::count(),
            'total_pos_revenue' => PosSale::sum('price') ?? 0,
        ];

        return response()->json($stats);
    }

    /**
     * List users
     */
    public function indexUsers(Request $request)
    {
        $users = User::paginate(10);

        return response()->json($users);
    }

    /**
     * Show user detail
     */
    public function showUser($id)
    {
        $user = User::with(['subscriptions', 'payments', 'posSales'])->findOrFail($id);

        return response()->json($user);
    }



    /**
     * Update user
     */
    public function updateUser(Request $request, $id)
    {
        $user = User::findOrFail($id);

        $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'email' => 'sometimes|required|string|email|max:255|unique:users,email,' . $id,
            'phone' => 'nullable|string|max:20',
            'is_admin' => 'boolean',
            'is_affiliate' => 'boolean',
            'affiliate_code' => 'nullable|string|size:8|unique:users,affiliate_code,' . $id,
        ]);

        // Handle affiliate code
        if ($request->has('is_affiliate')) {
            if ($request->is_affiliate) {
                // Activating affiliate
                if (!$user->is_affiliate) {
                    // New affiliate - generate or use provided code
                    if ($request->filled('affiliate_code')) {
                        $user->affiliate_code = Str::upper($request->affiliate_code);
                    } else {
                        // Generate unique random code
                        do {
                            $code = Str::upper(Str::random(8));
                        } while (User::where('affiliate_code', $code)->exists());
                        $user->affiliate_code = $code;
                    }
                } else {
                    // Already affiliate - allow code update if provided
                    if ($request->filled('affiliate_code')) {
                        $user->affiliate_code = Str::upper($request->affiliate_code);
                    }
                }
            } else {
                // Deactivating affiliate
                $user->affiliate_code = null;
            }
        }

        $user->update($request->only(['name', 'email', 'phone', 'is_admin', 'is_affiliate']));

        return response()->json($user);
    }

    /**
     * Delete user
     */
    public function destroyUser($id)
    {
        $user = User::findOrFail($id);
        $user->delete();

        return response()->json(['message' => 'User deleted']);
    }

    /**
     * List transactions
     */
    public function indexTransactions(Request $request)
    {
        $transactions = Payment::with(['user', 'subscription'])->paginate(10);

        return response()->json($transactions);
    }

    /**
     * Show transaction detail
     */
    public function showTransaction($id)
    {
        $transaction = Payment::with(['user', 'subscription', 'plan'])->findOrFail($id);

        return response()->json($transaction);
    }

    /**
     * List subscriptions
     */
    public function indexSubscriptions(Request $request)
    {
        $subscriptions = Subscription::with(['user', 'plan'])->paginate(10);

        return response()->json($subscriptions);
    }

    /**
     * Show subscription detail
     */
    public function showSubscription($id)
    {
        $subscription = Subscription::with(['user', 'plan', 'payments'])->findOrFail($id);

        return response()->json($subscription);
    }

    /**
     * Get subscription stats for dashboard
     */
    public function subscriptionStats()
    {
        $stats = [
            'active' => Subscription::where('status', 'active')->count(),
            'expired' => Subscription::where('status', 'expired')->count(),
            'cancelled' => Subscription::where('status', 'cancelled')->count(),
        ];

        return response()->json($stats);
    }

    /**
     * Get user growth stats (monthly)
     */
    public function userGrowthStats()
    {
        $stats = User::selectRaw('MONTH(created_at) as month, YEAR(created_at) as year, COUNT(*) as count')
            ->groupBy('year', 'month')
            ->orderBy('year', 'asc')
            ->orderBy('month', 'asc')
            ->get()
            ->map(function ($item) {
                return [
                    'month' => $item->month,
                    'year' => $item->year,
                    'count' => $item->count,
                ];
            });

        return response()->json($stats);
    }

    /**
     * Get revenue growth stats (monthly)
     */
    public function revenueGrowthStats()
    {
        $stats = Payment::selectRaw('MONTH(created_at) as month, YEAR(created_at) as year, SUM(amount) as revenue')
            ->groupBy('year', 'month')
            ->orderBy('year', 'asc')
            ->orderBy('month', 'asc')
            ->get()
            ->map(function ($item) {
                return [
                    'month' => $item->month,
                    'year' => $item->year,
                    'revenue' => $item->revenue,
                ];
            });

        return response()->json($stats);
    }

    /**
     * Get affiliate settings
     */
    public function getAffiliateSettings()
    {
        return response()->json(AffiliateSetting::getSettings());
    }

    /**
     * Update affiliate settings
     */
    public function updateAffiliateSettings(Request $request)
    {
        $request->validate([
            'commission_percentage' => 'required|numeric|min:0|max:100',
            'discount_percentage' => 'required|numeric|min:0|max:100',
        ]);

        $settings = AffiliateSetting::firstOrCreate([]);
        $settings->update($request->only(['commission_percentage', 'discount_percentage']));

        return response()->json($settings);
    }

    /**
     * List affiliate withdraws
     */
    public function getAffiliateWithdraws(Request $request)
    {
        $withdraws = AffiliateWithdraw::with('user')->paginate(10);

        return response()->json($withdraws);
    }

    /**
     * Approve withdraw
     */
    public function approveWithdraw($id)
    {
        $withdraw = AffiliateWithdraw::findOrFail($id);
        $withdraw->update(['status' => 'approved']);

        return response()->json(['message' => 'Withdraw approved']);
    }

    /**
     * Reject withdraw
     */
    public function rejectWithdraw($id)
    {
        $withdraw = AffiliateWithdraw::findOrFail($id);
        $withdraw->update(['status' => 'rejected']);

        // Return balance to user
        $withdraw->user->increment('affiliate_balance', $withdraw->amount);

        return response()->json(['message' => 'Withdraw rejected']);
    }

    /**
     * List POS sales (admin view - all users)
     */
    public function indexPosSales(Request $request)
    {
        $query = PosSale::with(['user', 'customer', 'product', 'channel', 'payment', 'admin']);

        // Filters
        if ($request->has('user_id')) {
            $query->where('user_id', $request->user_id);
        }
        if ($request->has('date_from')) {
            $query->whereDate('date', '>=', $request->date_from);
        }
        if ($request->has('date_to')) {
            $query->whereDate('date', '<=', $request->date_to);
        }

        $posSales = $query->paginate(10);

        return response()->json($posSales);
    }

    /**
     * Show POS sale detail
     */
    public function showPosSale($id)
    {
        $posSale = PosSale::with(['user', 'customer', 'product', 'channel', 'payment', 'admin'])->findOrFail($id);

        return response()->json($posSale);
    }

    /**
     * Update POS sale
     */
    public function updatePosSale(Request $request, $id)
    {
        $posSale = PosSale::findOrFail($id);

        $request->validate([
            'pos_customer_id' => 'sometimes|required|exists:pos_customers,id',
            'pos_product_id' => 'sometimes|required|exists:pos_products,id',
            'pos_channel_id' => 'sometimes|required|exists:pos_channels,id',
            'pos_payment_id' => 'sometimes|required|exists:pos_payments,id',
            'pos_admin_id' => 'sometimes|required|exists:pos_admins,id',
            'price' => 'sometimes|required|numeric|min:0',
            'link' => 'nullable|string|max:255',
            'date' => 'sometimes|required|date',
            'ship_date' => 'nullable|date',
            'note' => 'nullable|string|max:1000',
        ]);

        $posSale->update($request->only([
            'pos_customer_id', 'pos_product_id', 'pos_channel_id', 'pos_payment_id', 'pos_admin_id',
            'price', 'link', 'date', 'ship_date', 'note'
        ]));

        return response()->json($posSale->load(['user', 'customer', 'product', 'channel', 'payment', 'admin']));
    }

    /**
     * Delete POS sale
     */
    public function destroyPosSale($id)
    {
        $posSale = PosSale::findOrFail($id);
        $posSale->delete();

        return response()->json(['message' => 'POS sale deleted']);
    }

    /**
     * Get POS sales stats (monthly revenue growth)
     */
    public function posSalesStats()
    {
        $stats = PosSale::selectRaw('MONTH(date) as month, YEAR(date) as year, SUM(price) as revenue')
            ->groupBy('year', 'month')
            ->orderBy('year', 'asc')
            ->orderBy('month', 'asc')
            ->get()
            ->map(function ($item) {
                return [
                    'month' => $item->month,
                    'year' => $item->year,
                    'revenue' => $item->revenue ?? 0,
                ];
            });

        return response()->json($stats);
    }

    /**
     * Get all plans (admin)
     */
    public function getPlans()
    {
        $plans = \App\Models\Plan::all();
        return response()->json($plans);
    }

    /**
     * Update plan (admin)
     */
    public function updatePlan(Request $request, $id)
    {
        $plan = \App\Models\Plan::findOrFail($id);

        $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'price' => 'sometimes|required|numeric|min:0',
            'discounted_price' => 'nullable|numeric|min:0',
            'features' => 'sometimes|array',
        ]);

        $plan->update($request->only(['name', 'price', 'discounted_price', 'features']));

        return response()->json($plan);
    }

    /**
     * Get global settings
     */
    public function getGlobalSettings()
    {
        $settings = GlobalSetting::first();
        if (!$settings) {
            $settings = GlobalSetting::create(['cs_phone' => '6281234567890']);
        }
        return response()->json($settings);
    }

    /**
     * Update global settings
     */
    public function updateGlobalSettings(Request $request)
    {
        $request->validate([
            'cs_phone' => 'nullable|string|max:20',
        ]);

        $settings = GlobalSetting::first();
        if (!$settings) {
            $settings = GlobalSetting::create(['cs_phone' => '6281234567890']);
        }

        $settings->update($request->only(['cs_phone']));

        return response()->json($settings);
    }
}
